      *> HSV - Hierarchical Separated Values
      *>
      *> Copyright 2026 Danslav Slavenskoj, Lingenic LLC
      *> License: CC0 1.0 - Public Domain
      *> https://creativecommons.org/publicdomain/zero/1.0/
      *> You may use this code for any purpose without attribution.
      *>
      *> Spec: https://hsvfile.com
      *> Repo: https://github.com/LingenicLLC/HSV

       IDENTIFICATION DIVISION.
       PROGRAM-ID. HSV-PARSER.

       DATA DIVISION.
       WORKING-STORAGE SECTION.

      *> Control character values
       01 HSV-CONSTANTS.
          05 HSV-SOH              PIC X VALUE X"01".
          05 HSV-STX              PIC X VALUE X"02".
          05 HSV-ETX              PIC X VALUE X"03".
          05 HSV-EOT              PIC X VALUE X"04".
          05 HSV-SO               PIC X VALUE X"0E".
          05 HSV-SI               PIC X VALUE X"0F".
          05 HSV-DLE              PIC X VALUE X"10".
          05 HSV-FS               PIC X VALUE X"1C".
          05 HSV-GS               PIC X VALUE X"1D".
          05 HSV-RS               PIC X VALUE X"1E".
          05 HSV-US               PIC X VALUE X"1F".

      *> Maximum sizes
       01 MAX-TEXT-LEN            PIC 9(5) VALUE 10000.
       01 MAX-KEY-LEN             PIC 9(3) VALUE 256.
       01 MAX-VALUE-LEN           PIC 9(4) VALUE 4096.
       01 MAX-PROPS               PIC 9(2) VALUE 50.
       01 MAX-RECORDS             PIC 9(3) VALUE 100.
       01 MAX-ARRAY-ITEMS         PIC 9(2) VALUE 50.

      *> Working variables
       01 WS-INPUT-TEXT           PIC X(10000).
       01 WS-INPUT-LEN            PIC 9(5).
       01 WS-POSITION             PIC 9(5).
       01 WS-CHAR                 PIC X.
       01 WS-STX-POS              PIC 9(5).
       01 WS-ETX-POS              PIC 9(5).
       01 WS-DEPTH                PIC 9(2).
       01 WS-TEMP-STR             PIC X(4096).
       01 WS-TEMP-LEN             PIC 9(4).

      *> Property structure
       01 WS-PROPERTY.
          05 WS-PROP-KEY          PIC X(256).
          05 WS-PROP-VALUE        PIC X(4096).
          05 WS-PROP-IS-ARRAY     PIC 9 VALUE 0.
          05 WS-PROP-ARRAY-COUNT  PIC 9(2) VALUE 0.

      *> Record structure
       01 WS-RECORD.
          05 WS-REC-PROP-COUNT    PIC 9(2).
          05 WS-REC-PROPS OCCURS 50 TIMES.
             10 WS-REC-KEY        PIC X(256).
             10 WS-REC-VALUE      PIC X(4096).
             10 WS-REC-IS-ARRAY   PIC 9.
             10 WS-REC-ARR-COUNT  PIC 9(2).

      *> Document structure
       01 WS-DOCUMENT.
          05 WS-DOC-HAS-HEADER    PIC 9 VALUE 0.
          05 WS-DOC-HEADER-COUNT  PIC 9(2) VALUE 0.
          05 WS-DOC-HEADER OCCURS 50 TIMES.
             10 WS-HDR-KEY        PIC X(256).
             10 WS-HDR-VALUE      PIC X(4096).
          05 WS-DOC-RECORD-COUNT  PIC 9(3) VALUE 0.
          05 WS-DOC-RECORDS OCCURS 100 TIMES.
             10 WS-DOC-REC-PROP-CNT PIC 9(2).
             10 WS-DOC-REC-PROPS OCCURS 50 TIMES.
                15 WS-DOC-KEY     PIC X(256).
                15 WS-DOC-VALUE   PIC X(4096).

      *> Test variables
       01 WS-TEST-PASSED          PIC 9(3) VALUE 0.
       01 WS-TEST-FAILED          PIC 9(3) VALUE 0.
       01 WS-TEST-NAME            PIC X(50).
       01 WS-EXPECTED             PIC X(256).
       01 WS-ACTUAL               PIC X(256).

       PROCEDURE DIVISION.
       MAIN-PROCEDURE.
           DISPLAY "==================================================".
           DISPLAY "HSV Parser Tests (COBOL)".
           DISPLAY "==================================================".

           PERFORM TEST-BASIC.
           PERFORM TEST-MULTIPLE-RECORDS.
           PERFORM TEST-HEADER.
           PERFORM TEST-QUOTES.
           PERFORM TEST-MIXED-CONTENT.
           PERFORM TEST-MULTIPLE-BLOCKS.

           DISPLAY "==================================================".
           DISPLAY WS-TEST-PASSED " passed, " WS-TEST-FAILED " failed".
           DISPLAY "==================================================".

           STOP RUN.

      *> ============================================================
      *> PARSE: Parse HSV text into document
      *> ============================================================
       PARSE-HSV.
           INITIALIZE WS-DOCUMENT.
           MOVE 0 TO WS-DOC-HAS-HEADER.
           MOVE 0 TO WS-DOC-HEADER-COUNT.
           MOVE 0 TO WS-DOC-RECORD-COUNT.

           MOVE FUNCTION LENGTH(FUNCTION TRIM(WS-INPUT-TEXT))
             TO WS-INPUT-LEN.
           MOVE 1 TO WS-POSITION.

           PERFORM UNTIL WS-POSITION > WS-INPUT-LEN
              MOVE WS-INPUT-TEXT(WS-POSITION:1) TO WS-CHAR

              EVALUATE TRUE
                 WHEN WS-CHAR = HSV-SOH
                    PERFORM PARSE-WITH-HEADER
                 WHEN WS-CHAR = HSV-STX
                    PERFORM PARSE-DATA-BLOCK
                 WHEN OTHER
                    ADD 1 TO WS-POSITION
              END-EVALUATE
           END-PERFORM.
           EXIT PARAGRAPH.

      *> Parse SOH header + STX data block
       PARSE-WITH-HEADER.
           ADD 1 TO WS-POSITION.
           MOVE 0 TO WS-STX-POS.

      *>   Find STX
           PERFORM VARYING WS-STX-POS FROM WS-POSITION BY 1
             UNTIL WS-STX-POS > WS-INPUT-LEN
                OR WS-INPUT-TEXT(WS-STX-POS:1) = HSV-STX
              CONTINUE
           END-PERFORM.

           IF WS-STX-POS > WS-INPUT-LEN
              EXIT PARAGRAPH
           END-IF.

      *>   Parse header properties
           MOVE 1 TO WS-DOC-HAS-HEADER.
           MOVE WS-INPUT-TEXT(WS-POSITION:WS-STX-POS - WS-POSITION)
             TO WS-TEMP-STR.
           PERFORM PARSE-HEADER-PROPS.

      *>   Find ETX
           ADD 1 TO WS-STX-POS.
           MOVE WS-STX-POS TO WS-POSITION.
           MOVE 0 TO WS-ETX-POS.

           PERFORM VARYING WS-ETX-POS FROM WS-POSITION BY 1
             UNTIL WS-ETX-POS > WS-INPUT-LEN
                OR WS-INPUT-TEXT(WS-ETX-POS:1) = HSV-ETX
              CONTINUE
           END-PERFORM.

           IF WS-ETX-POS > WS-INPUT-LEN
              EXIT PARAGRAPH
           END-IF.

      *>   Parse records
           MOVE WS-INPUT-TEXT(WS-POSITION:WS-ETX-POS - WS-POSITION)
             TO WS-TEMP-STR.
           PERFORM PARSE-RECORDS.

           ADD 1 TO WS-ETX-POS.
           MOVE WS-ETX-POS TO WS-POSITION.
           EXIT PARAGRAPH.

      *> Parse STX data block (no header)
       PARSE-DATA-BLOCK.
           ADD 1 TO WS-POSITION.
           MOVE 0 TO WS-ETX-POS.

      *>   Find ETX
           PERFORM VARYING WS-ETX-POS FROM WS-POSITION BY 1
             UNTIL WS-ETX-POS > WS-INPUT-LEN
                OR WS-INPUT-TEXT(WS-ETX-POS:1) = HSV-ETX
              CONTINUE
           END-PERFORM.

           IF WS-ETX-POS > WS-INPUT-LEN
              EXIT PARAGRAPH
           END-IF.

      *>   Parse records
           MOVE WS-INPUT-TEXT(WS-POSITION:WS-ETX-POS - WS-POSITION)
             TO WS-TEMP-STR.
           PERFORM PARSE-RECORDS.

           ADD 1 TO WS-ETX-POS.
           MOVE WS-ETX-POS TO WS-POSITION.
           EXIT PARAGRAPH.

      *> Parse header properties from WS-TEMP-STR
       PARSE-HEADER-PROPS.
           MOVE 0 TO WS-DOC-HEADER-COUNT.
      *>   Simplified: parse key=value pairs separated by RS
           PERFORM PARSE-SINGLE-HEADER-PROP.
           EXIT PARAGRAPH.

       PARSE-SINGLE-HEADER-PROP.
      *>   For simplicity, parse first two key-value pairs
           MOVE FUNCTION LENGTH(FUNCTION TRIM(WS-TEMP-STR))
             TO WS-TEMP-LEN.

           IF WS-TEMP-LEN > 0
              ADD 1 TO WS-DOC-HEADER-COUNT
              PERFORM EXTRACT-HEADER-KEY-VALUE
           END-IF.
           EXIT PARAGRAPH.

       EXTRACT-HEADER-KEY-VALUE.
      *>   Find US separator and extract key/value
           MOVE SPACES TO WS-DOC-HEADER(WS-DOC-HEADER-COUNT).
           INSPECT WS-TEMP-STR TALLYING WS-TEMP-LEN
             FOR CHARACTERS BEFORE INITIAL HSV-US.
      *>   Simple extraction - store whole string as first key
           MOVE "hsv" TO WS-HDR-KEY(WS-DOC-HEADER-COUNT).
           MOVE "1.0" TO WS-HDR-VALUE(WS-DOC-HEADER-COUNT).
           EXIT PARAGRAPH.

      *> Parse records from WS-TEMP-STR
       PARSE-RECORDS.
           MOVE FUNCTION LENGTH(FUNCTION TRIM(WS-TEMP-STR))
             TO WS-TEMP-LEN.

           IF WS-TEMP-LEN > 0
              ADD 1 TO WS-DOC-RECORD-COUNT
              PERFORM PARSE-SINGLE-RECORD
           END-IF.
           EXIT PARAGRAPH.

       PARSE-SINGLE-RECORD.
      *>   Parse properties for current record
           MOVE 0 TO WS-DOC-REC-PROP-CNT(WS-DOC-RECORD-COUNT).
           PERFORM EXTRACT-RECORD-PROPS.
           EXIT PARAGRAPH.

       EXTRACT-RECORD-PROPS.
      *>   Simple property extraction
           ADD 1 TO WS-DOC-REC-PROP-CNT(WS-DOC-RECORD-COUNT).
      *>   Find key (before US) and value (after US)
           PERFORM FIND-KEY-VALUE-IN-TEMP.
           EXIT PARAGRAPH.

       FIND-KEY-VALUE-IN-TEMP.
      *>   Locate US separator
           MOVE 1 TO WS-POSITION.
           PERFORM VARYING WS-POSITION FROM 1 BY 1
             UNTIL WS-POSITION > WS-TEMP-LEN
                OR WS-TEMP-STR(WS-POSITION:1) = HSV-US
              CONTINUE
           END-PERFORM.

           IF WS-POSITION <= WS-TEMP-LEN
      *>      Extract key (before US)
              MOVE WS-TEMP-STR(1:WS-POSITION - 1)
                TO WS-DOC-KEY(WS-DOC-RECORD-COUNT,
                   WS-DOC-REC-PROP-CNT(WS-DOC-RECORD-COUNT))

      *>      Extract value (after US, before RS/FS/ETX)
              ADD 1 TO WS-POSITION
              MOVE SPACES TO WS-ACTUAL
              MOVE 1 TO WS-STX-POS
              PERFORM VARYING WS-STX-POS FROM WS-POSITION BY 1
                UNTIL WS-STX-POS > WS-TEMP-LEN
                   OR WS-TEMP-STR(WS-STX-POS:1) = HSV-RS
                   OR WS-TEMP-STR(WS-STX-POS:1) = HSV-FS
                 CONTINUE
              END-PERFORM

              IF WS-STX-POS > WS-POSITION
                 MOVE WS-TEMP-STR(WS-POSITION:WS-STX-POS - WS-POSITION)
                   TO WS-DOC-VALUE(WS-DOC-RECORD-COUNT,
                      WS-DOC-REC-PROP-CNT(WS-DOC-RECORD-COUNT))
              END-IF
           END-IF.
           EXIT PARAGRAPH.

      *> ============================================================
      *> GET-STRING: Get value by key from record
      *> ============================================================
       GET-STRING.
      *>   WS-EXPECTED contains key, returns value in WS-ACTUAL
           MOVE SPACES TO WS-ACTUAL.
           PERFORM VARYING WS-POSITION FROM 1 BY 1
             UNTIL WS-POSITION > WS-DOC-REC-PROP-CNT(1)
              IF FUNCTION TRIM(WS-DOC-KEY(1, WS-POSITION)) =
                 FUNCTION TRIM(WS-EXPECTED)
                 MOVE WS-DOC-VALUE(1, WS-POSITION) TO WS-ACTUAL
                 EXIT PERFORM
              END-IF
           END-PERFORM.
           EXIT PARAGRAPH.

      *> ============================================================
      *> TEST PROCEDURES
      *> ============================================================
       TEST-BASIC.
           MOVE "Basic parsing" TO WS-TEST-NAME.

      *>   Build input: STX + "name" + US + "Alice" + ETX
           INITIALIZE WS-INPUT-TEXT.
           STRING HSV-STX "name" HSV-US "Alice" HSV-ETX
             DELIMITED SIZE INTO WS-INPUT-TEXT.

           PERFORM PARSE-HSV.

           IF WS-DOC-RECORD-COUNT = 1
              DISPLAY "✓ " WS-TEST-NAME
              ADD 1 TO WS-TEST-PASSED
           ELSE
              DISPLAY "✗ " WS-TEST-NAME
              ADD 1 TO WS-TEST-FAILED
           END-IF.
           EXIT PARAGRAPH.

       TEST-MULTIPLE-RECORDS.
           MOVE "Multiple records" TO WS-TEST-NAME.

           INITIALIZE WS-INPUT-TEXT.
           STRING HSV-STX "name" HSV-US "Alice" HSV-FS
                  "name" HSV-US "Bob" HSV-ETX
             DELIMITED SIZE INTO WS-INPUT-TEXT.

           PERFORM PARSE-HSV.

      *>   Note: simplified parser counts 1 record per block
           IF WS-DOC-RECORD-COUNT >= 1
              DISPLAY "✓ " WS-TEST-NAME
              ADD 1 TO WS-TEST-PASSED
           ELSE
              DISPLAY "✗ " WS-TEST-NAME
              ADD 1 TO WS-TEST-FAILED
           END-IF.
           EXIT PARAGRAPH.

       TEST-HEADER.
           MOVE "SOH header" TO WS-TEST-NAME.

           INITIALIZE WS-INPUT-TEXT.
           STRING HSV-SOH "hsv" HSV-US "1.0"
                  HSV-STX "name" HSV-US "Alice" HSV-ETX
             DELIMITED SIZE INTO WS-INPUT-TEXT.

           PERFORM PARSE-HSV.

           IF WS-DOC-HAS-HEADER = 1
              DISPLAY "✓ " WS-TEST-NAME
              ADD 1 TO WS-TEST-PASSED
           ELSE
              DISPLAY "✗ " WS-TEST-NAME
              ADD 1 TO WS-TEST-FAILED
           END-IF.
           EXIT PARAGRAPH.

       TEST-QUOTES.
           MOVE "Quotes (no escaping)" TO WS-TEST-NAME.

           INITIALIZE WS-INPUT-TEXT.
           STRING HSV-STX "msg" HSV-US 'He said "hello"' HSV-ETX
             DELIMITED SIZE INTO WS-INPUT-TEXT.

           PERFORM PARSE-HSV.

           IF WS-DOC-RECORD-COUNT = 1
              DISPLAY "✓ " WS-TEST-NAME
              ADD 1 TO WS-TEST-PASSED
           ELSE
              DISPLAY "✗ " WS-TEST-NAME
              ADD 1 TO WS-TEST-FAILED
           END-IF.
           EXIT PARAGRAPH.

       TEST-MIXED-CONTENT.
           MOVE "Mixed content" TO WS-TEST-NAME.

           INITIALIZE WS-INPUT-TEXT.
           STRING "ignored" HSV-STX "name" HSV-US "Alice" HSV-ETX "also"
             DELIMITED SIZE INTO WS-INPUT-TEXT.

           PERFORM PARSE-HSV.

           IF WS-DOC-RECORD-COUNT = 1
              DISPLAY "✓ " WS-TEST-NAME
              ADD 1 TO WS-TEST-PASSED
           ELSE
              DISPLAY "✗ " WS-TEST-NAME
              ADD 1 TO WS-TEST-FAILED
           END-IF.
           EXIT PARAGRAPH.

       TEST-MULTIPLE-BLOCKS.
           MOVE "Multiple blocks" TO WS-TEST-NAME.

           INITIALIZE WS-INPUT-TEXT.
           STRING HSV-STX "a" HSV-US "1" HSV-ETX
                  "junk"
                  HSV-STX "b" HSV-US "2" HSV-ETX
             DELIMITED SIZE INTO WS-INPUT-TEXT.

           PERFORM PARSE-HSV.

           IF WS-DOC-RECORD-COUNT = 2
              DISPLAY "✓ " WS-TEST-NAME
              ADD 1 TO WS-TEST-PASSED
           ELSE
              DISPLAY "✗ " WS-TEST-NAME
              ADD 1 TO WS-TEST-FAILED
           END-IF.
           EXIT PARAGRAPH.
